package application;

import javafx.fxml.FXML;
import javafx.scene.control.Button;
import javafx.scene.control.Label;
import javafx.scene.control.ListView;
import javafx.scene.control.TextField;
import javafx.collections.FXCollections;
import javafx.collections.ObservableList;
import javafx.collections.ListChangeListener;
import java.util.stream.Collectors;

public class MainController {
	
	@FXML
	private ListView<String> listViewItems;
	
	@FXML
	private TextField txtNouvelItem;
	
	@FXML
	private TextField txtRecherche;
	
	@FXML
	private Button btnAjouter;
	
	@FXML
	private Button btnSupprimer;
	
	@FXML
	private Button btnFiltrer;
	
	@FXML
	private Button btnTrier;
	
	@FXML
	private Button btnSupprimerDoublons;
	
	@FXML
	private Button btnRemplacerTout;
	
	@FXML
	private Button btnMajuscules;
	
	@FXML
	private Label lblNombreItems;
	
	@FXML
	private Label lblMessage;
	
	// Liste complète (tous les éléments)
	private ObservableList<String> itemsComplets;
	
	// Liste filtrée (affichée dans la ListView)
	private ObservableList<String> items;
	
	@FXML
	private void initialize() {
		// Liste complète avec éléments initiaux
		itemsComplets = FXCollections.observableArrayList(
			"Pomme", "Banane", "Orange", "Fraise", "Cerise", "Kiwi", "Mangue"
		);
		
		// Liste filtrée (affichée dans la ListView)
		// Initialement, elle contient tous les éléments
		items = FXCollections.observableArrayList(itemsComplets);
		listViewItems.setItems(items);
		
		// Ajouter un listener pour écouter les changements
		items.addListener(new ListChangeListener<String>() {
			@Override
			public void onChanged(Change<? extends String> change) {
				while (change.next()) {
					if (change.wasAdded()) {
						lblMessage.setText("Éléments ajoutés : " + change.getAddedSubList());
						lblMessage.getStyleClass().setAll("message-success");
					}
					if (change.wasRemoved()) {
						lblMessage.setText("Éléments supprimés : " + change.getRemoved());
						lblMessage.getStyleClass().setAll("message-warning");
					}
					if (change.wasReplaced()) {
						lblMessage.setText("Liste remplacée");
						lblMessage.getStyleClass().setAll("message-success");
					}
				}
				
				// Mettre à jour le compteur automatiquement
				lblNombreItems.setText("Nombre d'éléments affichés : " + items.size() + 
				                      " / " + itemsComplets.size() + " au total");
			}
		});
		
		// Initialiser le compteur
		lblNombreItems.setText("Nombre d'éléments affichés : " + items.size() + 
		                      " / " + itemsComplets.size() + " au total");
		lblMessage.setText("Prêt à utiliser");
		
		// Écouter les changements dans le champ de recherche
		txtRecherche.textProperty().addListener((obs, oldVal, newVal) -> {
			// Filtrer automatiquement quand le texte change
			filtrer();
		});
	}
	
	@FXML
	private void ajouterItem() {
		String nouvelItem = txtNouvelItem.getText().trim();
		
		if (!nouvelItem.isEmpty()) {
			// Ajouter à la liste complète
			itemsComplets.add(nouvelItem);
			
			// Si l'élément correspond au filtre actuel, l'ajouter aussi à la liste affichée
			String recherche = txtRecherche.getText().toLowerCase().trim();
			if (recherche.isEmpty() || nouvelItem.toLowerCase().contains(recherche)) {
				items.add(nouvelItem);
			}
			
			txtNouvelItem.clear();
			lblMessage.setText("Élément ajouté : " + nouvelItem);
			lblMessage.getStyleClass().setAll("message-success");
		} else {
			lblMessage.setText("Veuillez saisir un élément");
			lblMessage.getStyleClass().setAll("message-warning");
		}
	}
	
	@FXML
	private void supprimerItem() {
		String selectionne = listViewItems.getSelectionModel().getSelectedItem();
		
		if (selectionne != null) {
			// Supprimer des deux listes
			items.remove(selectionne);
			itemsComplets.remove(selectionne);
			
			lblMessage.setText("Élément supprimé : " + selectionne);
			lblMessage.getStyleClass().setAll("message-warning");
		} else {
			lblMessage.setText("Veuillez sélectionner un élément à supprimer");
			lblMessage.getStyleClass().setAll("message-warning");
		}
	}
	
	@FXML
	private void filtrer() {
		String recherche = txtRecherche.getText().toLowerCase().trim();
		
		if (recherche.isEmpty()) {
			// Afficher tous les éléments
			items.setAll(itemsComplets);
			lblMessage.setText("Filtre supprimé - Affichage de tous les éléments");
			lblMessage.getStyleClass().setAll("message-success");
		} else {
			// Filtrer selon le critère
			items.setAll(
				itemsComplets.stream()
					.filter(item -> item.toLowerCase().contains(recherche))
					.collect(Collectors.toList())
			);
			
			lblMessage.setText("Filtre appliqué : " + recherche + " (" + items.size() + " résultat(s))");
			lblMessage.getStyleClass().setAll("message-success");
		}
	}
	
	@FXML
	private void trier() {
		// Trier la liste affichée
		FXCollections.sort(items);
		
		// Trier aussi la liste complète pour cohérence
		FXCollections.sort(itemsComplets);
		
		// Réappliquer le filtre si nécessaire
		if (!txtRecherche.getText().trim().isEmpty()) {
			filtrer();
		}
		
		lblMessage.setText("Liste triée par ordre alphabétique");
		lblMessage.getStyleClass().setAll("message-success");
	}
	
	@FXML
	private void supprimerDoublons() {
		// Créer une nouvelle liste sans doublons pour la liste complète
		itemsComplets.setAll(
			itemsComplets.stream()
				.distinct()
				.collect(Collectors.toList())
		);
		
		// Réappliquer le filtre pour mettre à jour la liste affichée
		filtrer();
		
		lblMessage.setText("Doublons supprimés");
		lblMessage.getStyleClass().setAll("message-success");
	}
	
	@FXML
	private void remplacerTout() {
		// Remplacer tout le contenu avec de nouveaux éléments
		itemsComplets.setAll("Nouveau 1", "Nouveau 2", "Nouveau 3", "Nouveau 4");
		
		// Réappliquer le filtre
		filtrer();
		
		lblMessage.setText("Liste remplacée avec de nouveaux éléments");
		lblMessage.getStyleClass().setAll("message-success");
	}
	
	@FXML
	private void mettreEnMajuscules() {
		// Utiliser replaceAll pour transformer tous les éléments
		itemsComplets.replaceAll(element -> element.toUpperCase());
		
		// Réappliquer le filtre
		filtrer();
		
		lblMessage.setText("Tous les éléments mis en majuscules");
		lblMessage.getStyleClass().setAll("message-success");
	}
	
	@FXML
	private void ajouterPlusieurs() {
		// Utiliser addAll pour ajouter plusieurs éléments en une seule opération
		itemsComplets.addAll("Ananas", "Pêche", "Abricot");
		
		// Réappliquer le filtre
		filtrer();
		
		lblMessage.setText("Plusieurs éléments ajoutés en une fois");
		lblMessage.getStyleClass().setAll("message-success");
	}
	
	@FXML
	private void supprimerConditionnel() {
		// Utiliser removeIf pour supprimer conditionnellement
		// Supprimer tous les éléments de moins de 5 caractères
		itemsComplets.removeIf(element -> element.length() < 5);
		
		// Réappliquer le filtre
		filtrer();
		
		lblMessage.setText("Éléments de moins de 5 caractères supprimés");
		lblMessage.getStyleClass().setAll("message-warning");
	}
	
	@FXML
	private void viderListe() {
		// Vider complètement la liste
		itemsComplets.clear();
		items.clear();
		
		lblMessage.setText("Liste vidée");
		lblMessage.getStyleClass().setAll("message-warning");
	}
	
	@FXML
	private void reinitialiser() {
		// Réinitialiser avec les éléments initiaux
		itemsComplets.setAll(
			"Pomme", "Banane", "Orange", "Fraise", "Cerise", "Kiwi", "Mangue"
		);
		
		// Réappliquer le filtre
		filtrer();
		
		lblMessage.setText("Liste réinitialisée");
		lblMessage.getStyleClass().setAll("message-success");
	}
}

