package application;

import javafx.fxml.FXML;
import javafx.scene.control.Label;
import javafx.scene.control.TextArea;
import javafx.stage.DirectoryChooser;
import javafx.stage.Stage;

import java.io.File;

public class MainController {
	
	@FXML
	private Label lblDossier; // Label pour afficher le dossier sélectionné
	
	@FXML
	private TextArea textArea; // TextArea pour afficher le contenu
	
	@FXML
	private Label lblMessage; // Label pour afficher les messages
	
	private Stage stagePrincipal;
	private File dossierCourant; // Dossier actuellement sélectionné
	
	/**
	 * Sélectionne un dossier avec DirectoryChooser
	 */
	@FXML
	private void selectionnerDossier() {
		DirectoryChooser directoryChooser = new DirectoryChooser();
		directoryChooser.setTitle("Sélectionner un dossier");
		directoryChooser.setInitialDirectory(new File(System.getProperty("user.home")));
		
		File dossierSelectionne = directoryChooser.showDialog(stagePrincipal);
		
		if (dossierSelectionne != null) {
			dossierCourant = dossierSelectionne;
			String chemin = dossierCourant.getAbsolutePath();
			lblDossier.setText("Dossier sélectionné : " + chemin);
			lblMessage.setText("Dossier sélectionné avec succès !");
			lblMessage.getStyleClass().setAll("message-success");
			textArea.clear();
			System.out.println("Dossier sélectionné : " + chemin);
		} else {
			lblMessage.setText("Aucun dossier sélectionné");
			lblMessage.getStyleClass().setAll("message-warning");
		}
	}
	
	/**
	 * Liste tout le contenu du dossier sélectionné
	 */
	@FXML
	private void listerContenu() {
		if (dossierCourant == null) {
			lblMessage.setText("Veuillez d'abord sélectionner un dossier");
			lblMessage.getStyleClass().setAll("message-warning");
			return;
		}
		
		try {
			StringBuilder contenu = new StringBuilder();
			contenu.append("=== CONTENU DU DOSSIER ===\n");
			contenu.append("Chemin : ").append(dossierCourant.getAbsolutePath()).append("\n\n");
			
			File[] fichiers = dossierCourant.listFiles();
			if (fichiers != null && fichiers.length > 0) {
				contenu.append("DOSSIERS :\n");
				for (File fichier : fichiers) {
					if (fichier.isDirectory()) {
						contenu.append("  📁 ").append(fichier.getName()).append("\n");
					}
				}
				
				contenu.append("\nFICHIERS :\n");
				for (File fichier : fichiers) {
					if (fichier.isFile()) {
						long taille = fichier.length();
						String tailleStr = taille < 1024 ? taille + " o" : 
										  taille < 1048576 ? (taille / 1024) + " Ko" : 
										  (taille / 1048576) + " Mo";
						contenu.append("  📄 ").append(fichier.getName())
							   .append(" (").append(tailleStr).append(")\n");
					}
				}
				contenu.append("\nTotal : ").append(fichiers.length).append(" élément(s)");
			} else {
				contenu.append("Le dossier est vide");
			}
			
			textArea.setText(contenu.toString());
			lblMessage.setText("Contenu listé avec succès");
			lblMessage.getStyleClass().setAll("message-success");
			
		} catch (Exception e) {
			lblMessage.setText("Erreur lors de la lecture : " + e.getMessage());
			lblMessage.getStyleClass().setAll("message-error");
			System.err.println("Erreur : " + e.getMessage());
			e.printStackTrace();
		}
	}
	
	/**
	 * Liste uniquement les images du dossier sélectionné
	 */
	@FXML
	private void listerImages() {
		if (dossierCourant == null) {
			lblMessage.setText("Veuillez d'abord sélectionner un dossier");
			lblMessage.getStyleClass().setAll("message-warning");
			return;
		}
		
		try {
			StringBuilder images = new StringBuilder();
			images.append("=== IMAGES TROUVÉES ===\n");
			images.append("Dossier : ").append(dossierCourant.getName()).append("\n\n");
			
			String[] extensions = {".png", ".jpg", ".jpeg", ".gif", ".bmp", ".webp"};
			File[] fichiers = dossierCourant.listFiles();
			
			if (fichiers != null) {
				int compteur = 0;
				for (File fichier : fichiers) {
					if (fichier.isFile()) {
						String nom = fichier.getName().toLowerCase();
						for (String ext : extensions) {
							if (nom.endsWith(ext)) {
								long taille = fichier.length();
								String tailleStr = taille < 1024 ? taille + " o" : 
												  taille < 1048576 ? (taille / 1024) + " Ko" : 
												  (taille / 1048576) + " Mo";
								images.append("🖼️  ").append(fichier.getName())
									  .append(" (").append(tailleStr).append(")\n");
								compteur++;
								break;
							}
						}
					}
				}
				images.append("\nTotal : ").append(compteur).append(" image(s)");
			}
			
			textArea.setText(images.toString());
			lblMessage.setText("Images listées avec succès");
			lblMessage.getStyleClass().setAll("message-success");
			
		} catch (Exception e) {
			lblMessage.setText("Erreur : " + e.getMessage());
			lblMessage.getStyleClass().setAll("message-error");
			System.err.println("Erreur : " + e.getMessage());
			e.printStackTrace();
		}
	}
	
	/**
	 * Crée des fichiers d'exemple dans le dossier sélectionné
	 */
	@FXML
	private void creerFichiers() {
		if (dossierCourant == null) {
			lblMessage.setText("Veuillez d'abord sélectionner un dossier");
			lblMessage.getStyleClass().setAll("message-warning");
			return;
		}
		
		try {
			// Créer plusieurs fichiers d'exemple
			File fichier1 = new File(dossierCourant, "exemple_texte.txt");
			File fichier2 = new File(dossierCourant, "exemple_data.txt");
			File fichier3 = new File(dossierCourant, "exemple_info.txt");
			
			java.nio.file.Files.write(fichier1.toPath(), 
				"Ceci est un fichier d'exemple créé par l'application JavaFX.".getBytes());
			java.nio.file.Files.write(fichier2.toPath(), 
				"Données d'exemple : ligne 1\nDonnées d'exemple : ligne 2".getBytes());
			java.nio.file.Files.write(fichier3.toPath(), 
				("Informations :\n- Fichier créé automatiquement\n- Date : " + 
				new java.util.Date().toString()).getBytes());
			
			lblMessage.setText("3 fichiers créés avec succès dans le dossier sélectionné !");
			lblMessage.getStyleClass().setAll("message-success");
			
			// Actualiser l'affichage
			listerContenu();
			
			System.out.println("Fichiers créés dans : " + dossierCourant.getAbsolutePath());
			
		} catch (Exception e) {
			lblMessage.setText("Erreur lors de la création : " + e.getMessage());
			lblMessage.getStyleClass().setAll("message-error");
			System.err.println("Erreur : " + e.getMessage());
			e.printStackTrace();
		}
	}
	
	public void setStagePrincipal(Stage stage) {
		this.stagePrincipal = stage;
	}
}

